// @ts-nocheck
import { diff } from 'deep-diff';
import { readFileSync, writeFileSync, mkdirSync, rmSync, readdirSync } from 'fs';
import { join } from 'path';
import { section_selectors, sectionSelectorLanguages, ai_section_selectors, aiSectionSelectorsLanguages, diffBaselineId } from '../constant/section_selectors';

const useAiSelectors = process.env.USE_AI_SELECTORS === 'true';

// Directory containing navigation JSON outputs to compare
const srcDir = 'src/detect_section_selector_masters/extractNavOutput';
// Directory to write nav diff results
const outDir = useAiSelectors ? 'src/detect_section_selector_masters/diffAINavWithBaseline' : 'src/detect_section_selector_masters/diffNavWithBaseline';

// Remove existing output folder if it exists
rmSync(outDir, { recursive: true, force: true });
// Ensure output directory exists
mkdirSync(outDir, { recursive: true });

const selectedLanguages = useAiSelectors ? aiSectionSelectorsLanguages : sectionSelectorLanguages;

// Baseline nav file
const baselineFile = `${diffBaselineId}_extractNav.json`;
const baselinePath = join(srcDir, baselineFile);
const baselineData = JSON.parse(readFileSync(baselinePath, 'utf-8'));

// Fields to ignore during diff
const ignoreFields = ['category', 'subcategory', 'id',];

// Get all nav files excluding baseline
const allFiles = readdirSync(srcDir).filter(f => f.endsWith('_extractNav.json') && f !== baselineFile);

allFiles.forEach(file => {
  const otherId = file.split('_')[0];
  const otherPath = join(srcDir, file);
  const otherData = JSON.parse(readFileSync(otherPath, 'utf-8'));

  // Compute differences
  const changes = diff(baselineData, otherData, (path, key) => ignoreFields.includes(key)) || [];

  // Convert to semantic format
  const semantic = changes.map(change => {
    const map: Record<string, string> = { E: '修改', N: '新增', D: '删除', A: '数组类型的变更' };
    const type = map[change.kind] || change.kind;
    const pathStr = (change.path || []).filter(seg => typeof seg !== 'number').join('.');
    let oldV: any, newV: any;
    if (change.kind === 'E') {
      oldV = (change as any).lhs;
      newV = (change as any).rhs;
    } else if (change.kind === 'N') {
      oldV = (change as any).lhs;
      newV = (change as any).rhs;
    } else if (change.kind === 'D') {
      oldV = (change as any).lhs;
      newV = (change as any).rhs;
    } else if (change.kind === 'A') {
      oldV = (change as any).item.lhs;
      newV = (change as any).item.rhs;
    }
    // Include category context for troubleshooting
    let categoryA: any = 'Not Applicable';
    let categoryB: any = 'Not Applicable';
    let subcategoryA: any = 'Not Applicable';
    let subcategoryB: any = 'Not Applicable';
    let idA: any = 'Not Applicable';
    let idB: any = 'Not Applicable';
    let typeA: any = 'Not Applicable';
    let typeB: any = 'Not Applicable';
    if (change.path && change.path[0] !== undefined && typeof change.path[0] === 'number') {
      const idx = change.path[0] as number;
      const entryA = baselineData[idx] || {};
      const entryB = otherData[idx] || {};
      categoryA = entryA.category || 'Not Applicable';
      categoryB = entryB.category || 'Not Applicable';
      subcategoryA = entryA.subcategory || 'Not Applicable';
      subcategoryB = entryB.subcategory || 'Not Applicable';
      idA = entryA.id || 'Not Applicable';
      idB = entryB.id || 'Not Applicable';
      typeA = entryA.type || 'Not Applicable';
      typeB = entryB.type || 'Not Applicable';
    }
    return {
      [`更改类型:${selectedLanguages[diffBaselineId]} vs ${selectedLanguages[otherId]}`]: type,
      "字段类型": pathStr,
      [`模版值:${selectedLanguages[diffBaselineId]}`]: oldV || 'Not Applicable',
      [`模版值:${selectedLanguages[otherId]}`]: newV || 'Not Applicable',
      [`category:${selectedLanguages[diffBaselineId]}`]: categoryA,
      [`category:${selectedLanguages[otherId]}`]: categoryB,
      [`subcategory:${selectedLanguages[diffBaselineId]}`]: subcategoryA,
      [`subcategory:${selectedLanguages[otherId]}`]: subcategoryB,
      [`type:${selectedLanguages[diffBaselineId]}`]: typeA,
      [`type:${selectedLanguages[otherId]}`]: typeB,
      [`id:${selectedLanguages[diffBaselineId]}`]: idA,
      [`id:${selectedLanguages[otherId]}`]: idB,
    };
  });

  // Write diff output
  const outFile = `${selectedLanguages[diffBaselineId]}_vs_${selectedLanguages[otherId]}_nav_diff.json`;
  const outPath = join(outDir, outFile);
  writeFileSync(outPath, JSON.stringify(semantic, null, 2), 'utf-8');
  console.log(`✅ Nav diff between ${diffBaselineId} and ${otherId} written to ${outPath}`);
});
