/**
 * AWS S3 文件上传 client
 *
 * 使用示例:
 * ```typescript
 * import { uploadToS3 } from './clients/s3';
 *
 * // 上传文件（自动读取环境变量）
 * const key = await uploadToS3(
 *   '/path/to/file.pdf',
 *   'uploads/file.pdf',
 *   { contentType: 'application/pdf', acl: 'private' }
 * );
 * ```
 */

import { S3Client, PutObjectCommand } from '@aws-sdk/client-s3';
import { readFileSync } from 'fs';

/**
 * S3 Configuration
 */
export interface S3Config {
  region: string;
  accessKeyId: string;
  secretAccessKey: string;
  bucketName: string;
}

/**
 * Upload options
 */
export interface UploadOptions {
  contentType?: string;
  acl?: 'private' | 'public-read';
}

/**
 * Create S3 client instance
 */
function createS3Client(config: S3Config): S3Client {
  return new S3Client({
    region: config.region,
    credentials: {
      accessKeyId: config.accessKeyId,
      secretAccessKey: config.secretAccessKey,
    },
  });
}

/**
 * Upload a file to S3
 * @param filePath Local path to the file
 * @param key S3 object key (path within bucket)
 * @param config S3 configuration
 * @param options Upload options
 * @returns The S3 object key
 */
export async function uploadFile(
  filePath: string,
  key: string,
  config: S3Config,
  options?: UploadOptions
): Promise<string> {
  const client = createS3Client(config);
  const fileContent = readFileSync(filePath);

  const uploadParams: any = {
    Bucket: config.bucketName,
    Key: key,
    Body: fileContent,
  };

  if (options?.contentType) {
    uploadParams.ContentType = options.contentType;
  }

  if (options?.acl) {
    uploadParams.ACL = options.acl;
  }

  const command = new PutObjectCommand(uploadParams);
  const response = await client.send(command);

  if (response.$metadata.httpStatusCode !== 200) {
    throw new Error(`S3 upload failed: ${response.$metadata.httpStatusCode}`);
  }

  return key;
}

/**
 * Upload a file to S3 (自动读取环境变量)
 * @param filePath Local path to the file
 * @param key S3 object key (path within bucket)
 * @param options Upload options
 * @returns The S3 object key
 */
export async function uploadToS3(
  filePath: string,
  key: string,
  options?: UploadOptions
): Promise<string> {
  return uploadFile(filePath, key, createS3ConfigFromEnv(), options);
}

/**
 * Create S3 config from environment variables
 * - AWS_REGION: AWS region (e.g., ap-northeast-1)
 * - AWS_ACCESS_KEY_ID: AWS access key ID
 * - AWS_SECRET_ACCESS_KEY: AWS secret access key
 * - AWS_S3_BUCKET_NAME: S3 bucket name
 */
export function createS3ConfigFromEnv(): S3Config {
  const region = process.env.AWS_REGION;
  const accessKeyId = process.env.AWS_ACCESS_KEY_ID;
  const secretAccessKey = process.env.AWS_SECRET_ACCESS_KEY;
  const bucketName = process.env.AWS_S3_BUCKET_NAME;

  if (!region || !accessKeyId || !secretAccessKey || !bucketName) {
    throw new Error(
      'Missing required S3 environment variables. Please set: AWS_REGION, AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY, AWS_S3_BUCKET_NAME'
    );
  }

  return {
    region,
    accessKeyId,
    secretAccessKey,
    bucketName,
  };
}
