import { sendSlackMessage } from '../clients/slack';
import { readdirSync, readFileSync } from 'fs';
import { join } from 'path';
import { diffBaselineId } from '../constant/section_selectors';

async function main() {
  // Scan diff directories for JSON files, with error handling
  const dirs = [
    join(__dirname, 'diffWithBaseline'),
    join(__dirname, 'diffAIWithBaseline'),
    join(__dirname, 'diffNavWithBaseline'),
    join(__dirname, 'diffAINavWithBaseline')
  ];
  let filePaths: string[] = [];
  dirs.forEach(dir => {
    try {
      const jsFiles = readdirSync(dir).filter(f => f.endsWith('.json'));
      jsFiles.forEach(f => filePaths.push(join(dir, f)));
    } catch (e) {
      console.warn(`Directory ${dir} not found, skipping.`);
    }
  });
  // Determine if any diff file contains non-empty array data
  const hasChanges = filePaths.some(fp => {
    try {
      const data = JSON.parse(readFileSync(fp, 'utf-8'));
      return Array.isArray(data) && data.length > 0;
    } catch {
      return false;
    }
  });

  // Check detectOutput file for diffBaselineId
  const detectOutputFile = join(__dirname, 'detectOutput', `${diffBaselineId}_detect.json`);
  let hasDetectChanges = false;
  let detectMessage = '';
  
  try {
    const detectData = JSON.parse(readFileSync(detectOutputFile, 'utf-8'));
    hasDetectChanges = Array.isArray(detectData) && detectData.length > 0;
    if (hasDetectChanges) {
      detectMessage = '⚠️ Issues are detected in the EN master section template.';
    }
  } catch (e) {
    console.warn(`DetectOutput file ${detectOutputFile} not found or invalid, skipping baseline check.`);
  }

  if (!hasChanges && !hasDetectChanges) {
    console.log('No changes detected in diff directories and baseline, skipping Slack notification.');
    process.exit(0);
  }

  const webhookUrl = process.env.SLACK_WEBHOOK!;
  const nonAiUrl = `https://docs.google.com/spreadsheets/d/${process.env.GOOGLE_SHEET_ID}/edit`;
  const aiUrl = `https://docs.google.com/spreadsheets/d/${process.env.GOOGLE_SHEET_AI_ID}/edit`;
  
  await sendSlackMessage(webhookUrl, {
    note: `${detectMessage}`,
    issue: 'Inconsistencies are detected in i18n master section template',
    details: `Google Sheet Links:\n- Non-AI selectors: ${nonAiUrl}\n- AI selectors: ${aiUrl}`
  });
}

main().catch(err => {
  console.error(err);
  process.exit(1);
});
